﻿using System;

using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.Threading;
using System.IO.Ports;
using MettlerToledo.IND780.SharedDataClient;
using MettlerToledo.IND780.Softkeys;


namespace Classify_Application
{
	public partial class Form3 : Form
	{
		private SDclient SD_client = ClassifyMain.SD_client;

		private Softkeys mySoftkeys;

		//public delegate void myCallbackDelegate(string sdname, string sdvalue);
		//public delegate void myCallbackGroupDelegate(int groupNumber, string sdValues);

		public static Form3 myForm3 = null;

		private string thresholdColorsSDlist = "ax0101 ax0102 ax0103 ax0104 ax0105 ax0106 ax0107 ax0108 ax0109 ax0110 ax0111 ax0112";
		private string thresholdWeightsSDlist = "ay0101 ay0102 ay0103 ay0104 ay0105 ay0106 ay0107 ay0108 ay0109 ay0110 ay0111 ay0112";
		
		private Color[] thresholdColors = {Color.White, Color.Yellow, Color.Blue, Color.Navy, Color.Red, Color.Maroon,
										  Color.Orange, Color.Purple, Color.Lime, Color.Green, Color.Pink, Color.LightGray,
										  Color.DarkGray, Color.Magenta, Color.Olive, Color.Teal};

		private Color[] textColors = {Color.Black, Color.Black, Color.White, Color.White, Color.Black, Color.White,
										 Color.Black, Color.White, Color.Black, Color.White, Color.Black, Color.Black,
										 Color.White, Color.Black, Color.White, Color.White};

		private double[] thresholdWeights = new double[12];
		private int[] colorOffsets = new int[12];

		private bool errorDisplaying = false;

		private int secondsCounter = 0;

		private bool serialPort1Open = false;
		private bool serialPort2Open = false;

		private bool displayCompressed = false;

		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form3::Form3
		* Description   :Constructor
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		public Form3()
		{
			myForm3 = this;

			InitializeComponent();

			if (SD_client.GetSharedDataConnectionStatus)		// display lime green if connected to SD server
			{
				string scaleSelected = SD_client.ReadSharedData("xt0101").TrimEnd();
				this.label_connected.BackColor = Color.Lime;
				this.RegisterClassifyCallbacks();
				this.RegisterClassifyGroupCallbacks();
				this.LoadThresholdColorsAndWeights();
				this.label_scale.Text = SD_client.ReadSharedData("cs0" + scaleSelected + "03");
				// RGROUP command to read time and IND780 on time
				SD_client.SetReadGroup(2, "xd0104 xp0113");
				this.timer2.Enabled = true;
				this.textBox_tare.Visible = false;
			}

			this.SoftkeysInitialization();

			this.RegisterSoftkeys();
			this.RegisterScaleKeys();

			this.TestSerialPorts();
		}

		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form3::SoftkeysInitialization
		* Description   :Creates the five picture boxes and draws the softkey borders used by the softkeys
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		private void SoftkeysInitialization()
		{
			mySoftkeys = new Softkeys();
			mySoftkeys.AllocateSoftkeys(this, 195, @"Storage Card\Terminal\SKBMP\COLOR\");
		}

		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form3::RegisterSoftkeys
		* Description   :Registers the softkey graphic and method used by each softkey.
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		private void RegisterSoftkeys()
		{
			// level 0 softkeys
			mySoftkeys.LoadSoftkeys(0, 0, true, @"Storage Card\Terminal\SKBMP\COLOR\exit.bmp", (Softkeys.mySKMethodDelegate)ClassifyWeighExit);
			mySoftkeys.LoadSoftkeys(0, 2, true, @"Storage Card\Terminal\SKBMP\COLOR\total.bmp", (Softkeys.mySKMethodDelegate)ClassifyCompressDisplay);
			mySoftkeys.LoadSoftkeys(0, 3, true, @"Storage Card\Terminal\SKBMP\COLOR\stop.bmp", (Softkeys.mySKMethodDelegate)ClassifyWeighStop);
			mySoftkeys.LoadSoftkeys(0, 4, true, @"Storage Card\Terminal\SKBMP\COLOR\report.bmp", (Softkeys.mySKMethodDelegate)ClassifyReport);

			mySoftkeys.NumberOfLevels = 1;

			mySoftkeys.DisplaySoftKeys(0);
		}

		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form3::RegisterScaleKeys
		* Description   :Registers the methods used by the zero and tare scale keys.
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		private void RegisterScaleKeys()
		{
			mySoftkeys.LoadApplicationAndScaleKeys(4, true, (Softkeys.mySKMethodDelegate)myApplicationMethodForScaleSelection);
			mySoftkeys.LoadApplicationAndScaleKeys(5, true, (Softkeys.mySKMethodDelegate)myApplicationMethodForZeroScale);
			mySoftkeys.LoadApplicationAndScaleKeys(6, true, (Softkeys.mySKMethodDelegate)myApplicationMethodForTareScale);
			mySoftkeys.LoadApplicationAndScaleKeys(7, true, (Softkeys.mySKMethodDelegate)myApplicationMethodForPrintScale);
		}

		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form3::myApplicationMethodForZeroScale
		* Description   :Zero scale key handler
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		private void myApplicationMethodForZeroScale()
		{
			string zStat = "";
			if (!SD_client.Zero(ref zStat))
			{
				this.DisplayError(zStat);
			}
			this.label_status.Text = ClassifyMain.GetString(516);	// "Pushbutton Zero"
			this.timer1.Interval = 1000;
			this.timer1.Enabled = true;
		}

		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form3::myApplicationMethodForTareScale
		* Description   :Tare scale key handler
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		private void myApplicationMethodForTareScale()
		{
			string tStat = "";
			if (this.textBox_tare.Text.Length == 0)
			{
				if (!SD_client.TarePB(ref tStat))
				{
					this.DisplayError(tStat);
					return;
				}
				this.label_status.Text = ClassifyMain.GetString(359);	// "Pushbutton Tare"
				this.timer1.Interval = 1000;
				this.timer1.Enabled = true;
			}
			else
			{
				if (!SD_client.TarePreset(this.textBox_tare.Text, ref tStat))
				{
					this.DisplayError(tStat);
					return;
				}
				this.label_status.Text = ClassifyMain.GetString(360);	// "Keyboard Tare"
				this.timer1.Interval = 1000;
				this.timer1.Enabled = true;
			}
		}

		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form3::myApplicationMethodForPrintScale
		* Description   :Print scale key handler
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		private void myApplicationMethodForPrintScale()
		{
			string pStat = "";
			if (!SD_client.Print(ref pStat))
			{
				this.DisplayError(pStat);
			}
			else
			{
				this.label_status.Text = ClassifyMain.GetString(2000);	// Printing
				this.timer1.Interval = 5000;
				this.timer1.Enabled = true;								// Display for 5 seconds
			}
		}

		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form3::myApplicationMethodForScaleSelection
		* Description   :Select scale key handler
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		private void myApplicationMethodForScaleSelection()
		{
			string ssStat = "";
			if (!SD_client.SelectScale(this.textBox_tare.Text, ref ssStat))
			{
				this.DisplayError(ssStat);
			}
			else
			{
				string selScl = SD_client.ReadSharedData("xt0101");
				this.textBox_tare.Text = "";
				this.label_scale.Text = SD_client.ReadSharedData("cs0" + selScl + "03");	// Get scale ID
				this.label_status.Text = "Scale Select";
				this.timer1.Interval = 1000;
				this.timer1.Enabled = true;
			}
		}

		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form3::ClearKey
		* Description   :Clear scale key handler
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		private void ClearKey()
		{
			string selected = SD_client.ReadSharedData("xt0101");
			int mode = int.Parse(SD_client.ReadSharedData("wx0" + selected + "35"));	// scale mode
			string cStat = "";

			if (mode == 1)	// net weight mode
			{
				if (!SD_client.Clear(ref cStat))
				{
					this.DisplayError(cStat);
				}
				else
				{
					this.textBox_tare.Text = "";
					this.label_status.Text = ClassifyMain.GetString(843);	// "Clear Tare"
					this.timer1.Interval = 1000;
					this.timer1.Enabled = true;
				}
			}
		}

		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form3::DisplayError
		* Description   :Pops up a error based on the errorStrin
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		private void DisplayError(string errorString)
		{
			this.label_error_header.Text = ClassifyMain.GetString(728);
			this.label_error_body.Text = errorString;
			this.panel_error.Visible = true;
			this.panel_error.BringToFront();	// to overlay TextBox_tare
			this.errorDisplaying = true;
		}


		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form3::ClassifyWeighExit
		* Description   :Delegate method that exits this form
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		private void ClassifyWeighExit()
		{
			// !!! Callbacks and timers must be terminated before closing the form or an exception will occur !!!
			SD_client.UnregisterCallbacks();
			SD_client.UnregisterCallbackGroups();
			this.timer1.Enabled = false;
			this.timer2.Enabled = false;
			this.Close();
		}

		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form3::ClassifyWeighStop
		* Description   :Delegate method that stops the classify operation and exits this form
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		private void ClassifyWeighStop()
		{
			// !!! Callbacks and timers must be terminated before closing the form or an exception will occur !!!
			SD_client.UnregisterCallbacks();
			SD_client.UnregisterCallbackGroups();
			this.timer1.Enabled = false;
			this.timer2.Enabled = false;
			//this.Close();
		}

		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form1::ClassifyReport
		* Description   :Sends out a simple Classify report when the Report softkey is pressed.
		*				:  Description	Weight		Date		Time
		*				:  Class 1		119.2 kg	27/Feb/2012	15:56:10
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		private void ClassifyReport()
		{
			string date = SD_client.ReadSharedData("xd0103");
			string time = SD_client.ReadSharedData("xd0104");
			string header = "Description" + "\t" + "Weight" + "\t\t" + "Date" + "\t\t" + "Time" + "\r\n";
			string wtData = this.label_Class.Text + "\t\t" + this.label_weight.Text + " " + this.label_units.Text + 
				"\t" + date + "\t" + time + "\r\n";
			SerialPort myPort = null;

			if (this.serialPort1Open)
				myPort = ClassifyMain.serialPort_1;
			else if (this.serialPort2Open)
				myPort = ClassifyMain.serialPort_2;
			else
				return;

			try
			{
				myPort.WriteLine(header);
				myPort.WriteLine(wtData);
			}
			catch
			{
				MessageBox.Show("Could not write class data");
			}
		}

		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form3::ClassifyCompressDisplay
		* Description   :Delegate method that compresses and uncompresses display
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		private void ClassifyCompressDisplay()
		{
			int bottomYcoordinate = 0;
			if (!this.displayCompressed)
			{
				bottomYcoordinate = SD_client.DisplayWt(SDclient.Display.Compress);
				this.displayCompressed = true;
			}
			else
			{
				bottomYcoordinate = SD_client.DisplayWt(SDclient.Display.Uncompress);
				this.displayCompressed = false;
			}
			this.label_displayMode.Text = " Bottom Y = " + bottomYcoordinate.ToString();
		}

		#region Callback methods
		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form3::RegisterClassifyCallbacks
		* Description   :Registers callbacks for weight, center of zero, and gross/net shared data
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		void RegisterClassifyCallbacks()
		{
			// Scale 1
			SD_client.RegisterCallback((SDclient.MyCallbackDelegate)ClassifyWeight_Callback, "wt0102");
			SD_client.RegisterCallback((SDclient.MyCallbackDelegate)ClassifyZero_Callback, "wx0132");
			SD_client.RegisterCallback((SDclient.MyCallbackDelegate)ClassifyGrossNet_Callback, "wx0135");
			SD_client.RegisterCallback((SDclient.MyCallbackDelegate)ClassifyMotion_Callback, "wx0131");
			// Scale 2
			SD_client.RegisterCallback((SDclient.MyCallbackDelegate)ClassifyWeight_Callback, "wt0202");
			SD_client.RegisterCallback((SDclient.MyCallbackDelegate)ClassifyZero_Callback, "wx0232");
			SD_client.RegisterCallback((SDclient.MyCallbackDelegate)ClassifyGrossNet_Callback, "wx0235");
			SD_client.RegisterCallback((SDclient.MyCallbackDelegate)ClassifyMotion_Callback, "wx0231");
		}

		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form3::ClassifyWeight_Callback
		* Description   :Invokes the weight update callback delegate
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		void ClassifyWeight_Callback(string sdName, string sdValue)
		{
			try
			{
				//Invoke(new myCallbackDelegate(WeightUpdate), new object[] { sdName, sdValue });
				Invoke(new SDclient.MyCallbackDelegate(WeightUpdate), new object[] { sdName, sdValue });
			}
			catch
			{
			}
		}

		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form3::ClassifyZero_Callback
		* Description   :Invokes the zero callback delegate
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		void ClassifyZero_Callback(string sdName, string sdValue)
		{
			try
			{
				//Invoke(new Form3.myCallbackDelegate(ZeroUpdate), new object[] { sdName, sdValue });
				Invoke(new SDclient.MyCallbackDelegate(ZeroUpdate), new object[] { sdName, sdValue });
			}
			catch
			{
			}
		}

		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form3::ClassifyGrossNet_Callback
		* Description   :Invokes the gross/net weight callback delegate
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		void ClassifyGrossNet_Callback(string sdName, string sdValue)
		{
			try
			{
				//Invoke(new Form3.myCallbackDelegate(GrossNetUpdate), new object[] { sdName, sdValue });
				Invoke(new SDclient.MyCallbackDelegate(GrossNetUpdate), new object[] { sdName, sdValue });
			}
			catch
			{
			}
		}

		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form3::ClassifyMotion_Callback
		* Description   :Invokes the weight motion callback delegate
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		void ClassifyMotion_Callback(string sdName, string sdValue)
		{
			try
			{
				//Invoke(new Form3.myCallbackDelegate(MotionUpdate), new object[] { sdName, sdValue });
				Invoke(new SDclient.MyCallbackDelegate(MotionUpdate), new object[] { sdName, sdValue });
			}
			catch
			{
			}
		}

		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form3::WeightUpdate
		* Description   :Updates the weight display and background and foreground colors
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		private void WeightUpdate(string sdName, string sdValue)
		{
			double sclWt = 0.0;
			string selected = SD_client.ReadSharedData("xt0101").TrimEnd();

			if (selected != sdName.Substring(3, 1))
				return;

			this.label_weight.Text = sdValue.TrimStart();
			sclWt = double.Parse(this.label_weight.Text);

			for (int i = 0; i < this.thresholdWeights.Length; i++ )
			{
				if (sclWt < this.thresholdWeights[i])
				{
					this.label_weight.BackColor = this.thresholdColors[this.colorOffsets[i]];
					this.label_weight.ForeColor = this.textColors[this.colorOffsets[i]];
					this.label_zero.BackColor = this.thresholdColors[this.colorOffsets[i]];
					this.label_zero.ForeColor = this.textColors[this.colorOffsets[i]];
					this.label_units.BackColor = this.thresholdColors[this.colorOffsets[i]];
					this.label_units.ForeColor = this.textColors[this.colorOffsets[i]];
					this.label_BG.BackColor = this.thresholdColors[this.colorOffsets[i]];
					this.label_BG.ForeColor = this.textColors[this.colorOffsets[i]];
					this.label_scale.BackColor = this.thresholdColors[this.colorOffsets[i]];
					this.label_scale.ForeColor = this.textColors[this.colorOffsets[i]];
					this.label_motion.BackColor = this.thresholdColors[this.colorOffsets[i]];
					this.label_motion.ForeColor = this.textColors[this.colorOffsets[i]];
					this.label_Class.BackColor = this.thresholdColors[this.colorOffsets[i]];
					this.label_Class.ForeColor = this.textColors[this.colorOffsets[i]];
					this.label_Class.Text = "Class " + (i + 1).ToString();
					break;
				}
			}
		}

		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form3::ZeroUpdate
		* Description   :Updates the zero indicator for the selected scale
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		private void ZeroUpdate(string sdName, string sdValue)
		{
			string selected = SD_client.ReadSharedData("xt0101").TrimEnd();

			if (selected != sdName.Substring(3, 1))
				return;

			int zero = int.Parse(sdValue);

			if (zero == 0)
				this.label_zero.Visible = false;
			else
				this.label_zero.Visible = true;
		}

		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form3::GrossNetUpdate
		* Description   :Updates the Net or B/G indicator for the selected scale
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		private void GrossNetUpdate(string sdName, string sdValue)
		{
			string selected = SD_client.ReadSharedData("xt0101").TrimEnd();
			int language = int.Parse(SD_client.ReadSharedData("xs0115"));

			if (selected != sdName.Substring(3, 1))
				return;

			if (int.Parse(sdValue) == 1)
				this.label_BG.Text = ClassifyMain.GetString(869);	// "Net";
			else
				this.label_BG.Text = ClassifyMain.GetString(772);	// "B/G";

		}

		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form3::MotionUpdate
		* Description   :Updates the motion indicator for the selected scale
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		private void MotionUpdate(string sdName, string sdValue)
		{
			string selected = SD_client.ReadSharedData("xt0101").TrimEnd();

			if (selected != sdName.Substring(3, 1))
				return;

			int motion = int.Parse(sdValue);

			if (motion == 0)
				this.label_motion.Visible = false;
			else
				this.label_motion.Visible = true;
		}
		#endregion

		#region Group Callback Methods
		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form3::RegisterClassifyGroupCallbacks
		* Description   :Registers group callbacks for weight
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		void RegisterClassifyGroupCallbacks()
		{
			SD_client.RegisterCallbackGroup((SDclient.MyCallbackGroupDelegate)ClassifyWeight_CallbackGroup, 1, "wt0102 wt0202");
			
		}
		#endregion

		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form3::ClassifyWeight_CallbackGroup
		* Description   :Invokes the delegate method GroupWeightUpdate
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		void ClassifyWeight_CallbackGroup(int groupNumber, string sdValues)
		{
			try
			{
				Invoke(new SDclient.MyCallbackGroupDelegate(GroupWeightUpdate), new object[] { groupNumber, sdValues });
			}
			catch
			{
			}
		}

		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form3::GroupWeightUpdate
		* Description   :Updates the two scale weights section
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		private void GroupWeightUpdate(int groupNumber, string sdValues)
		{
			string[] weights = sdValues.Split('~');
			this.label_information.Text = weights[0] + " " + weights[1];
		}

		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form3::LoadThresholdColorsAndWeights
		* Description   :Loads the threshold colors and weights from shared data and stores them for use in
		*				:the weight update method.
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		void LoadThresholdColorsAndWeights()
		{
			string[] colorIdx = SD_client.ReadSharedDataList(this.thresholdColorsSDlist).Split('~');
			string[] thresholdWt = SD_client.ReadSharedDataList(this.thresholdWeightsSDlist).Split('~');

			for (int i = 0; i < this.colorOffsets.Length; i++)
			{
				try
				{
					this.colorOffsets[i] = int.Parse(colorIdx[i]);
				}
				catch
				{
					this.colorOffsets[i] = 0;		// white
				}

				try
				{
					this.thresholdWeights[i] = double.Parse(thresholdWt[i]);
				}
				catch
				{
					this.thresholdWeights[i] = 0.0;
				}
			}
		}

		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form3::Common_KeyUp
		* Description   :Common key-up event handler for all keys.  Calls the softkeys dispatcher
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		void Common_KeyUp(object sender, System.Windows.Forms.KeyEventArgs e)
		{
			if (!mySoftkeys.SoftkeyDispatcher((int)e.KeyCode))
				mySoftkeys.ApplicationAndScaleKeysDispatcher((int)e.KeyCode);

			if (e.KeyCode == Keys.Back)	// Clear key
			{
				this.ClearKey();
			}

			if (e.KeyCode == Keys.Enter)
			{
				if (this.errorDisplaying)
				{
					this.errorDisplaying = false;
					this.panel_error.Visible = false;
				}
			}
		}

		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form3::Common_KeyDown
		* Description   :Common key-down event handler for all keys.  Forces a key click noise.
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		void Common_KeyDown(object sender, System.Windows.Forms.KeyEventArgs e)
		{
			// There is just one textBox in this application (textBox_tare) and its visibility is initially
			// set to false.  This is done so that no blinking cursor is displayed.  The first numeric key
			// or decimal point will cause textBox_tare to become visible with the first entry placed at
			// position 0.  The cursor is then moved to position 1 (SelectStart = 1).
			int key = (int)(e.KeyCode);
			if ((key >= 96 && key <= 105) || key == 110)	// 0 - 9 and decimal point
			{
				this.textBox_tare.Visible = true;
				if (this.textBox_tare.Text.Length == 0 && !this.textBox_tare.Focused)
				{
					this.textBox_tare.Focus();
					if (key == 110)
						this.textBox_tare.Text = ".";
					else
						this.textBox_tare.Text = (key - 96).ToString();
					this.textBox_tare.SelectionStart = 1;
					this.textBox_tare.Refresh();
				}
			}

			ClassifyMain.SD_client.SetSharedData("xc0134", "1");	// key click
		}

		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form3::Form3_Paint
		* Description   :The ClassifyMain Paint event handler.
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		void Form3_Paint(object sender, System.Windows.Forms.PaintEventArgs e)
		{
			mySoftkeys.DrawSoftkeyBorders(this);
		}


		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form3::timer1_Tick
		* Description   :Timer1 interrupt handler
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		private void timer1_Tick(object sender, EventArgs e)
		{
			this.timer1.Enabled = false;
			this.label_status.Text = "";
		}

		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form3::timer2_Tick
		* Description   :Timer2 interrupt handler
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		private void timer2_Tick(object sender, EventArgs e)
		{
			string[] groupString = SD_client.ReadSharedDataGroup(2).Split('~');	// read xd0104 and xd0113

			this.timer2.Enabled = false;

			this.label_time.Text = groupString[0];			// xd0104 (time)

			if (this.secondsCounter == 0)
			{
				int hrs = 0;
				int mins = 0;
				try
				{
					mins = int.Parse(groupString[1]);
					hrs = mins / 60;
					mins = mins % 60;
				}
				catch
				{
				}
				this.label_ontime.Text = hrs.ToString() + ":" + mins.ToString("D2");
				this.secondsCounter = 60;
			}
			this.secondsCounter--;

			this.timer2.Enabled = true;
		}

		/*---------------------------------------------------------------------------------------------------------*
		* Name          :Form3::TestSerialPorts
		* Description   :Displays which serial ports are open
		* Return Value  :None
		* --------------------------------------------------------------------------------------------------------*/
		private void TestSerialPorts()
		{
			this.label_serial_status.Text = string.Empty;

			if (ClassifyMain.serialPort_1.PortName.Length >= 4)
			{
				if (ClassifyMain.serialPort_1.PortName == "COM1:" && ClassifyMain.serialPort_1.IsOpen)
				{
					this.label_serial_status.Text = "COM1 Open";
					this.serialPort1Open = true;
				}

				if (ClassifyMain.serialPort_2.PortName == "COM2:" && ClassifyMain.serialPort_2.IsOpen)
				{
					if (this.label_serial_status.Text == string.Empty)
						this.label_serial_status.Text = "COM2 Open";
					else
						this.label_serial_status.Text += "COM1-COM2 Open";
					this.serialPort2Open = true;
				}
			}
		}

	}
}